﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using System.Threading;
using System.Timers;


/* BackgroundWorker Class  */
namespace Lessons
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent(); 
        }

        BackgroundWorker worker;
        bool executing = false;

        /* this handler is not synchronized */
        void Worker_Work(object sender, DoWorkEventArgs e)
        {
            Command cmd = (Command)e.Argument;              // e.Argument is the object passed to the BackgroundWorker.RunWorkerAsync()      
            cmd.Execute((BackgroundWorker)sender,  e);

            /* e.Result will be passed to the RunWorkerCompletedEventArgs of the BackgroundWorker.RunWorkerCompleted event */
            e.Result = cmd;                                  
        }

        /* synchronized handler */
        void Worker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            Command cmd = (Command)e.UserState;         // e.UserState is an object passed to the BackgroundWorker.ReportProgress()
            progressBar1.Value = e.ProgressPercentage;  // same as above

            textBox1.Text = "worker progress: " + e.ProgressPercentage.ToString() 
                            + "%, Total size: " + cmd.DataSize.ToString() + " (" + cmd.RemainSize.ToString() + ")";
        }

        /* synchronized handler */
        void Worker_Completed(object sender, RunWorkerCompletedEventArgs e)
        {
            if (e.Error != null)
                textBox1.Text = e.Error.Message;
            else if (e.Cancelled)
                textBox1.Text += Environment.NewLine + "Canceled";
            else
                textBox1.Text += Environment.NewLine + "DONE";

            executing = false;
            progressBar1.Value = 0;
        }


        private void btnStart_Click(object sender, EventArgs e)
        {
            if (!executing)
            {
                textBox1.Text = "";                
                
                /* create a worker */
                worker = new BackgroundWorker();

                /* configure a worker */
                worker.WorkerReportsProgress = true;
                worker.WorkerSupportsCancellation = true;

                /* link worker events */
                worker.DoWork += new DoWorkEventHandler(Worker_Work);
                worker.ProgressChanged += new ProgressChangedEventHandler(Worker_ProgressChanged);
                worker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(Worker_Completed);
                
                /* start the worker */
                worker.RunWorkerAsync(new Command());    

                executing = true;
            }
        }


        private void btnStop_Click(object sender, EventArgs e)
        {
            if (executing)
            {
                if (worker.WorkerSupportsCancellation)
                {
                    worker.CancelAsync();   // this sets BackgroundWorker.CancellationPending to true
                    executing = false;
                }
            }
        }

        private void btnClear_Click(object sender, EventArgs e)
        {
            textBox1.Text = "";
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = executing;

            if (e.Cancel)
                MessageBox.Show("Please stop worker manually! Worker is still executed...");
        }
 
    }
}



namespace Lessons
{
    class Command
    {
        static Random random = new Random();

        private int dataSize = random.Next(30, 50);
        private int remainSize = 0;

        public void Execute(BackgroundWorker worker, DoWorkEventArgs e)
        {
            remainSize = dataSize;
            int donePercent = 0;

            while (remainSize > 0)
            {
                if (worker.CancellationPending) // BackgroundWorker.CancelAsync() has called
                {
                    e.Cancel = true;
                    break;
                }
                else
                {
                    remainSize--;
                    donePercent = ((dataSize - remainSize) * 100) / dataSize;
                    Thread.Sleep(150);

                    worker.ReportProgress(donePercent, this);  // feed the BackgroundWorker.ProgressChanged event 
                }
            }
        }

        public int DataSize { get { return dataSize; } }
        public int RemainSize { get { return remainSize; } }
    }

 
}